/*
 * CharacterObject.h
 *
 * Created 8/27/2009 By Johnny Huynh
 *
 * Version 00.00.01 8/27/2009
 *
 * Copyright Information:
 * All content copyright  2009 Johnny Huynh. All rights reserved.
 */
 
 #ifndef CHARACTER_OBJECT_H
 #define CHARACTER_OBJECT_H
 
 template <typename T> class CharacterObject;
 
 #include "CollisionActor.h"
 
 #include "global.h"
 
 /**
  * Class specification for CharacterObject
  */
 template <typename T>
 class CharacterObject : public CollisionActor<T>
 {
 // Data Members
 private:
    T _movement_speed;
    int _stun_counter;
 
 // Local Functions
 public:
    CharacterObject( const T movement_speed = ONE );
    CharacterObject( const CharacterObject<T>& character );
    virtual ~CharacterObject();
    inline CharacterObject<T>& operator=( const CharacterObject<T>& character );
    virtual inline const T& get_movement_speed() const;
    virtual inline bool is_mobile() const;
    virtual inline bool is_prone_to_effects() const;
    virtual inline bool is_stunned() const;
    virtual inline void stun();
    virtual inline void unstun();
    virtual inline void interrupt_action();
    
    // overloaded functions (NodePath)
    /*static void init_type() {
                                std::string template_type( typeid( T ).name() );
                                register_type(_type_handle, "CharacterObject<" + template_type + ">" );
                            }*/
 
 // Protected Functions
 protected:
    virtual inline void set_movement_speed( const T& movement_speed );
    
 // Static Functions
 public:
    //static inline void interrupt_action( CharacterObject<T>* character_Ptr );
    
 };
 
 /** LOCAL FUNCTIONS **/
 
 /**
  * Constructor
  */
 template <typename T>
 CharacterObject<T>::CharacterObject( const T movement_speed )
                    : CollisionActor<T>(),
                      _movement_speed( movement_speed ),
                      _stun_counter( 0 )
 {
    
 }
 
 /**
  * Copy Constructor
  */
 template <typename T>
 CharacterObject<T>::CharacterObject( const CharacterObject<T>& character )
                    : CollisionActor<T>( character ),
                      _movement_speed( character._movement_speed ),
                      _stun_counter( 0 )
 {
    
 }
 
 /**
  * Destructor
  */
 template <typename T>
 CharacterObject<T>::~CharacterObject()
 {
    
 }
 
 /**
  * operator=() copies the content of the specified CharacterObject to this CharacterObject.
  *
  * @param (const CharacterObject<T>&) character
  * @return CharacterObject<T>&
  */
 template <typename T>
 inline CharacterObject<T>& CharacterObject<T>::operator=( const CharacterObject<T>& character )
 {
    CollisionActor<T>::operator=( static_cast< CollisionActor<T> >( character ) );
    _movement_speed = character._movement_speed;
    _stun_counter = 0;
    
    return *this;
 }
 
 /**
  * get_movement_speed() returns the _movement_speed of this CharacterObject.
  *
  * @return const T&
  */
 template <typename T>
 inline const T& CharacterObject<T>::get_movement_speed() const
 {
    return _movement_speed;
 }
 
 /**
  * is_mobile() returns true if this CharacterObject can move in the world; otherwise, false is returned.
  *
  * @return bool
  */
 template <typename T>
 inline bool CharacterObject<T>::is_mobile() const
 {
    return _movement_speed != ZERO && !CharacterObject<T>::is_stunned();
 }
 
 /**
  * is_prone_to_effects() returns true if this Object is prone to 
  * physical or status effects; otherwise, false is returned.
  *
  * @return bool
  */
 template <typename T>
 inline bool CharacterObject<T>::is_prone_to_effects() const
 {
    return true;
 }
 
 /**
  * is_stunned() returns true if this CharacterObject is stunned; otherwise, false is returned.
  *
  * @return bool
  */
 template <typename T>
 inline bool CharacterObject<T>::is_stunned() const
 {
    return _stun_counter != 0;
 }
 
 /**
  * stun() indicates the addition of a stun state for this CharacterObject.
  */
 template <typename T>
 inline void CharacterObject<T>::stun()
 {
    if ( _stun_counter >= 0 )
        ++_stun_counter;
 }
 
 /**
  * unstun() indicates the removal of a stun state for this CharacterObject.
  */
 template <typename T>
 inline void CharacterObject<T>::unstun()
 {
    if ( _stun_counter > 0 )
        --_stun_counter;
 }
 
 /**
  * interrupt_action() removes all tasks in the global action task manager that are associated with this character
  * to stop this character from performing any action.
  */
 template <typename T>
 inline void CharacterObject<T>::interrupt_action()
 {
    Object<T>::clear_action_tasks();
    
    //assume_neutral_position();
    Actor<T>::get_anims().stop_all(); // stop all animations in their tracks
 }
 
 /** PROTECTED FUNCTIONS **/
 
 /**
  * set_movement_speed() sets the _movement_speed of this CharacterObject to
  * the specified movement_speed.
  *
  * @param (const T&) movement_speed
  */
 template <typename T>
 inline void CharacterObject<T>::set_movement_speed( const T& movement_speed )
 {
    _movement_speed = movement_speed;
 }
 
 /** STATIC FUNCTIONS **/
 
 /**
  * interrupt_action() removes all tasks in the global action task manager that are associated with this character
  * to stop this character from performing any action.
  *
  * @param (CharacterObject<T>*) character_Ptr
  */
 /*template <typename T>
 inline void CharacterObject<T>::interrupt_action( CharacterObject<T>* character_Ptr )
 {
    nassertv( character_Ptr != NULL );
    character_Ptr->Object<T>::clear_action_tasks();
    
    //character_Ptr->assume_neutral_position();
    character_Ptr->Actor<T>::get_anims().stop_all(); // stop all animations in their tracks
 }*/
 
 #endif // CHARACTER_OBJECT_H